;;; handwrite.el --- turns your emacs buffer into a handwritten document  -*- lexical-binding: t -*-

;; Copyright (C) 1996-2026 Free Software Foundation, Inc.

;; Author: Danny Roozendaal (was: <danny@tvs.kun.nl>)
;; Maintainer: emacs-devel@gnu.org
;; Created: October 21 1996
;; Keywords: text, print, postscript, cursive writing

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; The function `handwrite' creates PostScript output containing a
;; handwritten version of the current buffer.
;;
;; Other functions that may be useful are:
;;
;;     `handwrite-10pt': set the font size to 10 and find corresponding
;;                       values for the line spacing and the number of lines
;;                       on a page.
;;     `handwrite-11pt': which is similar
;;     `handwrite-12pt': which is also similar
;;     `handwrite-13pt': which is similar, too
;;
;;     `handwrite-set-pagenumber': set and unset page numbering
;;
;;
;; If you are not satisfied with the type page there are a number of
;; variables you may want to set.
;;
;; To use this, say `M-x handwrite' or type at your prompt
;; "emacs -l handwrite.el".
;;
;; I tried to make it `iso_8859_1'-friendly, but there are some exotic
;; characters missing.
;;
;;
;; Known bugs:
;; - Page feeds do not work, and are ignored instead.
;; - Tabs are not always properly displayed.
;; - Handwrite may create corrupt PostScript if it encounters
;;   unknown characters.
;;
;; Thanks to anyone who emailed me suggestions!

;;; Code:

(require 'ps-print)

;; Variables

(defgroup handwrite nil
  "Turn your Emacs buffer into a handwritten document."
  :prefix "handwrite-"
  :group 'games)

(defvar handwrite-psindex 0
  "The index of the PostScript buffer.")
(defvar menu-bar-handwrite-map
  (let ((map (make-sparse-keymap "Handwrite functions.")))
    (define-key map [numbering]
      '(menu-item "Page numbering" handwrite-set-pagenumber
        :button (:toggle . handwrite-pagenumbering)))
    (define-key map [handwrite-separator2] '("----" . nil))
    (define-key map [10pt] '(menu-item "10 pt" handwrite-10pt
                             :button (:radio . (eq handwrite-fontsize 10))))
    (define-key map [11pt] '(menu-item "11 pt" handwrite-11pt
                             :button (:radio . (eq handwrite-fontsize 11))))
    (define-key map [12pt] '(menu-item "12 pt" handwrite-12pt
                             :button (:radio . (eq handwrite-fontsize 12))))
    (define-key map [13pt] '(menu-item "13 pt" handwrite-13pt
                             :button (:radio . (eq handwrite-fontsize 13))))
    (define-key map [handwrite-separator1] '("----" . nil))
    (define-key map [handwrite] '("Write by hand" . handwrite))
    map))
(fset 'menu-bar-handwrite-map menu-bar-handwrite-map)
(make-obsolete 'menu-bar-handwrite-map nil "28.1")
(make-obsolete-variable 'menu-bar-handwrite-map nil "28.1")

;; User definable variables

(defcustom handwrite-numlines 60
  "The number of lines on a page of the PostScript output from `handwrite'."
  :type 'integer)

(defcustom handwrite-fontsize 11
  "The size of the font for the PostScript output from `handwrite'."
  :type 'integer)

(defcustom handwrite-linespace 12
  "The spacing for the PostScript output from `handwrite'."
  :type 'integer)

(defcustom handwrite-xstart 30
  "X-axis translation in the PostScript output from `handwrite'."
  :type 'integer)

(defcustom handwrite-ystart 810
  "Y-axis translation in the PostScript output from `handwrite'."
  :type 'integer)

(defcustom handwrite-pagenumbering nil
  "If non-nil, number each page of the PostScript output from `handwrite'."
  :type 'boolean)

(defcustom handwrite-10pt-numlines 65
  "The number of lines on a page for the function `handwrite-10pt'."
  :type 'integer)

(defcustom handwrite-11pt-numlines 60
  "The number of lines on a page for the function `handwrite-11pt'."
  :type 'integer)

(defcustom handwrite-12pt-numlines 55
  "The number of lines on a page for the function `handwrite-12pt'."
  :type 'integer)

(defcustom handwrite-13pt-numlines 50
  "The number of lines on a page for the function `handwrite-13pt'."
  :type 'integer)

;; Interactive functions

;;;###autoload
(defun handwrite ()
  "Turn the buffer into a \"handwritten\" document.
The functions `handwrite-10pt', `handwrite-11pt', `handwrite-12pt'
and `handwrite-13pt' set up for various sizes of output.

Variables: `handwrite-linespace'     (default 12)
           `handwrite-fontsize'      (default 11)
           `handwrite-numlines'      (default 60)
           `handwrite-pagenumbering' (default nil)"
  (interactive)
  (setq handwrite-psindex (1+ handwrite-psindex))
  (let
      ((cur-buf (current-buffer))
       (tpoint (point))
       (ps-ypos 63)
       (lcount 0)
       (ipage 1)
       (next-line-add-newlines t)
       (buf-name (buffer-name) )
       (textp)
       (ps-buf-name (format "*handwritten%d.ps*" handwrite-psindex))
       (trans-table
	'(("ÿ" . "264") ("á" . "207") ("à" . "210") ("â" . "211")
	  ("ä" . "212") ("ã" . "213") ("å" . "214") ("é" . "216")
	  ("è" . "217") ("ê" . "220") ("ë" . "221") ("í" . "222")
	  ("ì" . "223") ("î" . "224") ("ï" . "225") ("ó" . "227")
	  ("ò" . "230") ("ô" . "231") ("ö" . "232") ("õ" . "233")
	  ("ú" . "234") ("ù" . "235") ("û" . "236") ("ü" . "237")
	  ("ß" . "247") ("°" . "241") ("®" . "250") ("©" . "251")
	  ("ij" . "264") ("ç" . "215") ("§" . "244") ("ñ" . "226")
	  ("£" . "243")))
       (escape-table '("\\\\" "(" ")")) ; \\ comes first to not work
					; on inserted backslashes
       line)
    (goto-char (point-min))		;start at beginning
    (switch-to-buffer ps-buf-name)
    (handwrite-insert-header buf-name)
    (insert "%%Creator: GNU Emacs's handwrite version " emacs-version  "\n")
    (handwrite-insert-preamble)
    (handwrite-insert-info)
    (handwrite-insert-font)
    (setq textp (point))
    (insert "%%Page: 1 1\n")
    (insert "Hwjst\n")
    (insert "/Hwsave save def\n")
    (if handwrite-pagenumbering (insert "20 30 m\nxym(page 1)a\n"))
    (insert "44 63 m\n")
    (insert "xym( )a")
    (backward-char 3)
    (switch-to-buffer cur-buf)
    (goto-char (point-min))		;start at beginning
    (save-excursion
      (while (not (eobp))
	(setq line (thing-at-point 'line))
	(dolist (escape escape-table)
	  (setq line (replace-regexp-in-string escape
					       (concat "\\\\" escape) line)))
	(dolist (trans trans-table)
	  (setq line (replace-regexp-in-string (car trans)
					       (concat "\\\\" (cdr trans))
					       line)))
	(switch-to-buffer ps-buf-name)
	(insert (string-replace "\n" "" line))
	(message "write write write...")
	(setq ps-ypos (+ ps-ypos handwrite-linespace))
	(end-of-line)
	(insert "\n")
	(setq lcount (+ lcount 1))
	(when (= lcount handwrite-numlines)
	  (setq ipage (+ ipage 1))
	  (insert "0 0  m\n")
	  (insert "showpage exec Hwsave restore\n")
	  (insert "%%Page: " (number-to-string ipage) " "
		  (number-to-string ipage) "\n")
	  (insert "Hwjst\n")
	  (insert "/Hwsave save def\n")
	  (if handwrite-pagenumbering
	      (insert "20 30 m\nxym(page "
		      (number-to-string ipage) ")a\n"))
	  (setq ps-ypos 63)
	  (setq lcount 0))
	(insert "44 " (number-to-string ps-ypos) " m\n")
	(insert "xym( )a")
	(backward-char 3)
	(switch-to-buffer cur-buf)
	(forward-line 1)
	))
    (switch-to-buffer ps-buf-name)
    (forward-line 1)
    (insert " showpage exec Hwsave restore\n\n")
    (insert "%%Pages " (number-to-string ipage) " 0\n")
    (insert "%%EOF\n")
    ;;To avoid cumbersome code we simply ignore formfeeds
    (goto-char textp)
    (while (search-forward "\f" nil t)
      (replace-match "" nil t) )
    (untabify textp (point-max))	; this may result in strange tabs
    (when (y-or-n-p "Send this to the printer? ")
      (let* ((coding-system-for-write 'raw-text-unix)
	     (printer-name (or ps-printer-name printer-name))
             (lpr-printer-switch ps-printer-name-option)
             (print-region-function ps-print-region-function)
             (lpr-command ps-lpr-command))
        (lpr-print-region (point-min) (point-max) ps-lpr-switches nil)))
    (message "")
    (bury-buffer ())
    (switch-to-buffer cur-buf)
    (goto-char tpoint)))


(defun handwrite-set-pagenumber ()
  "Toggle the value of `handwrite-pagenumbering'."
  (interactive)
  (if handwrite-pagenumbering
      (handwrite-set-pagenumber-off)
    (handwrite-set-pagenumber-on)))

(defun handwrite-10pt ()
  "Specify 10-point output for `handwrite'.
Set `handwrite-fontsize' to 10 and find correct values for
`handwrite-linespace' and `handwrite-numlines'."
  (interactive)
  (setq handwrite-fontsize 10)
  (setq handwrite-linespace 11)
  (setq handwrite-numlines handwrite-10pt-numlines)
  (message "Handwrite output size set to 10 points"))

(defun handwrite-11pt ()
  "Specify 11-point output for `handwrite'.
Set `handwrite-fontsize' to 11 and find correct values for
`handwrite-linespace' and `handwrite-numlines'."
  (interactive)
  (setq handwrite-fontsize 11)
  (setq handwrite-linespace 12)
  (setq handwrite-numlines handwrite-11pt-numlines)
  (message "Handwrite output size set to 11 points"))

(defun handwrite-12pt ()
  "Specify 12-point output for `handwrite'.
Set `handwrite-fontsize' to 12 and find correct values for
`handwrite-linespace' and `handwrite-numlines'."
  (interactive)
  (setq handwrite-fontsize 12)
  (setq handwrite-linespace 13)
  (setq handwrite-numlines handwrite-12pt-numlines)
  (message "Handwrite output size set to 12 points"))

(defun handwrite-13pt ()
  "Specify 13-point output for `handwrite'.
Set `handwrite-fontsize' to 13 and find correct values for
`handwrite-linespace' and `handwrite-numlines'."
  (interactive)
  (setq handwrite-fontsize 13)
  (setq handwrite-linespace 14)
  (setq handwrite-numlines handwrite-13pt-numlines)
  (message "Handwrite output size set to 13 points"))


;; Internal Functions

;;The header for the PostScript output. The argument is the name of
;;the original buffer
(defun handwrite-insert-header (buf-name)
  (insert "%!PS-Adobe-2.0\n")
  (insert "%%Title: " buf-name "\n")
  (insert "%%CreationDate: " (current-time-string) "\n" )
  (insert "%%Pages: (atend)\n")
  (insert "%%For: "  user-mail-address   "\n")
  (insert "%%EndComments\n"))

;;Some PostScript definitions for using our font.
(defun handwrite-insert-preamble ()
  (insert "\n%%BeginPreamble
/m {moveto} def
/r {roll} def
/e {exch} def
/a {awidthshow} def
/xym {0.52490 0. 32 0.05249 0.} def
/HwJdict 80 dict def
/Hwfdict 80 dict def
/getsymb {Hwfdict /Jsymb get 3 1 r put} def
/Chread{
  {  e begin
      HwJdict begin %read in character specifications
        /jnum e def
        /jnum1 jnum 6 mul def
        save symbstr jnum1 6 getinterval{
        }forall
        /pixvol e def
        /pixwid e def
        /charwidth e def
        /trx e def
        /try e def
        /pixf e def
        .02666667 .02666667  scale
        /pixwid pixwid 1.042 mul def
        /pixwidn pixwid trx add def
        /pixvoln pixvol try add def
        charwidth 0 trx try pixwidn pixvoln setcachedevice
        newpath 0 0 m
        pixf 0 gt{
          pixf 3 bitshift
          trx try translate
          pixwid pixvol scale
          /pixvol1 {pixvol 4 add true} def
          /pixvol2 pixvol neg def
          /pixvol3 pixvol 2 add def
          pixvol1 pixwid 0 0 pixvol2 0 pixvol3 6 array astore
          Jsymb jnum get
          imagemask
        }if
        restore
      end
    end
  }def
}def
/Hwjst{
  /Joepie findfont [Hws 0 Hws pop 0 Hws neg 0 0] makefont setfont
}def
%%EndPreamble\n"))

;;The font size for the PostScript output.
;;Also the x-y-translations of the PostScript stuff.
(defun handwrite-insert-info ()
  (insert "\n%%BeginSizeTranslate\n")
  (insert "/Hws " (number-to-string handwrite-fontsize) " def")
  (insert "         %The size of the font\n")
  (insert (number-to-string handwrite-xstart)" "
	  (number-to-string handwrite-ystart))
  (insert " translate    %Translate the origin to the current location\n")
  (insert "1 -1 scale          %Flip the y coordinate\n")
  (insert "%%EndSizeTranslate\n\n"))


;;Bitmap PostScript font for pseudo handwritten documents.
;;Two years ago, I created the 36 points font for my
;;Apple Macintosh Classic II resulting in unusual ascii numbers.
(defun handwrite-insert-font ()
  (insert "\n%%BeginFont Joepie
Hwfdict begin
  /Jsymb 256 array  def
  /FontType 3 def
  /FontBBox [0 0 1 1] def
  /FontMatrix [1 0 0 1 0 0] def
  /Encoding 256 array def
end
<002809000000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 002809080000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 002809000000 FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 002809080000 020A0E08031E 021C0B0E0A0A
040C09161418 04050A121323 040D0F1C141B 040C141C0F19 021B0B08060D
02080B0C091E 0208080C0B20 041A0B100F0C 040E09141314 02060B08060B
0219090C0C03 020C0B080304 04090C120F1F 04080A121017 020C10120514
040A09121216 040C09121216 02090B120D15 040C0B121015 040A0A12121B
020209120D1D 040A09121018 04040A12101C 020C0D080412 020408080717
040B0B141116 040C0B16140B 040B0B141018 020A0D120C1C 04090B1C191F
040A091A1C1C 0408071A1E20 040A0B1A1A1E 040E0D201E1A 040C0916181C
040E0916171A 04010B1C1C27 060A0C20201E 040E060E141A 0401070E1327
040E0A1C1D1A 040C0916181C 060C07262B1C 060C0520261C 04080D1E1C20
040A05161C1C 040A0B1E1E1E 040A091A1D1E 040C0914171C 040E07161A1A
060E091E1F1A 040E071A1D1A 060A0B2C2C1E 040808181B20 040409161722
040C08181A1C 020A080A0A1D 040A0610121E 0208090A0A1F 041C0714160A
040505161903 021B0C0C060D 040C08141713 040C0916181C 040C08121412
040D0916181B 040C09141615 0202090C0D26 04040814171B 040A0916181E
020C090A0D1B 0202090A0C23 040A0814171E 020C080A0D1C 060C09222412
040C08161910 040C08141611 04000816191C 04020914171B 040A090E1014
040A09101213 040C090C0F1B 040C09161812 040A09121412 040C09181A10
040C08121511 04020812151C 040C08101312 040A050A0F1E 020A1114031C
0208070A0E20 04140A141409 FF28FFFF0000 040A071A1B1C 040A091A181C
040A0B1A171E 020F0F160E16 040A0F201A1E 04080D1E1A20 040E0B1E1D1A
040C0814171B 040D0714171B 040D0714171A 040D0814171A 040D0814171B
040D0814171B 04000912141C 040C0914151C 040C0914161C 040C0914151C
040C0914151B 020B070A0D1C 020B070A0D1C 020B080A0D1C 020B070A0D1A
040C0616191A 040B0714171D 040B08141719 040B0814171A 040B08141718
040B08141718 040B0716191B 040B0716191B 040B0716191C 040B08161919
040A0912111C 02180C0E090C 04050712141F 040A0816151C 04080A121220
020F0A120E10 040408181522 04040816191D 040A0D201A1E 040A09201C1C
06190B24210E 021C0B0C0708 0221090E0D03 040809141415 040C121E1619
04080F1E1920 04130F1A1309 040C0B141216 04090914141E 040A0914131B
040209161823 04030A161519 020B0B120E19 0407091A1921 040A0B1E141E
040C0914140F 0401090C0F25 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
FF28FFFF0000 040A0B14121A 04010910101B FF28FFFF0000 040A0614181C
040A09141618 0407090E1221 040D0A141510 002800160000 04080A141119
04080614151A 040A0B1E1804 FF28FFFF0000 040A071A1B1D 040A071A191D
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 041408141303 061409262404
041A0B120F0B 021D0B120D0A 021E0E0A050A 021E0B0A060A FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 040600060F1C FF28FFFF0000
020C0A0C0B16 020D080C0C17 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
02140B0A0506 02040908060E FF28FFFF0000 0608092A291E FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
040A0B1C181B 060C071E221C FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
FF28FFFF0000 021D0C0E0C08 021E090E0C06 0421060E1105 FF28FFFF0000
021F0A0A0506 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000 FF28FFFF0000
041E09101107>
Hwfdict begin /symbstr e def end
255< 00000000 00000000 80018000 60030000 38060000 0E1C0000 03F00000
01C00000 01800000 00000000 00000000>getsymb
250< 0000 0000 6000 7800 7800 F800 F800 7000 0000 0000>getsymb
248< 00000000 00000000 F8000000 3F800000 03F80000 007E0000 00038000
00000000 00000000 >getsymb
247< 0000 0000 0030 0060 38E0 7FC0 C300 8000 0000 0000 >getsymb
246< 0000 0000 0C00 1E00 3F00 3380 60C0 4040 C020 0010 0000 0000
>getsymb
241< 000000000000 000000000000 008000000000 00C000000000 03C000000000
038000000000 000000000000 000000000000 000000000000 000000000000
010000000000 010000000000 010007C00000 03001FE00000 070038700000
070030300000 0F0060300000 1B00C0700000 3300C0600000 6300C0C0C000
C300C3818000 030046030000 03007C020000 0300F0060000 0180F00C0000
018198180000 01C30C300000 00CE06300000 00FC03E00000 007801C00000
000000000000 000000000000 >getsymb
240< 00000000 00000000 00004000 0003C000 00038000 000E0000 000C0000
00080000 00180000 0E187C00 1F98FC00 3FDCFF00 3FFFFF00 7FFFFE00
7FFFFC00 FFFFF800 FFFFF000 FFFFF000 FFFFF000 FFFFF000 FFFFF000
FFFFFC00 7FFFFC00 7FFFFF00 3FFFFF00 1FFFFE00 0FFFFE00 07FFF800
03F3F000 00000000 00000000 >getsymb
228< 000000000000 000000000000 03C010000000 07C0F0000000 1C7330000000
383F20000000 303040000000 2030C0000000 6030C0000000 C01080000000
C01180000000 C03100000000 406300000000 604200000000 60C40E000080
71843F01E080 3F0CE183B180 1E0880CE1F00 001180FA0600 003100460600
0023004C0200 0047004C0300 004B004C0300 00B300CC0300 00E300CC0200
01C1818C0200 03C1830C0600 0380830C0C00 0600FE061800 1C007C03F000
180000000000 300000000000 000000000000 000000000000 >getsymb
226< 0000 0000 2000 1800 1800 0C00 0C00 0C00 0C00 0C00 0C00 1800 1000
3000 6000 8000 0000 0000 >getsymb
225< 0000 0000 7000 E800 F800 F800 F000 F000 0000 0000 >getsymb
221< 0000 0000 8000 E000 3000 1C00 0E00 0700 0380 01C0 01C0 00E0 0030
0030 0030 0060 00C0 0180 0300 0200 0600 0C00 1800 3000 4000 0000 0000
>getsymb
220< 0000 0000 0060 0040 00C0 0180 0300 0600 0C00 1C00 3000 7000 6000
C000 4000 6000 3000 3000 1800 1C00 0E00 0600 0300 0100 0000 0000
>getsymb
218< 00000000 00000000 00020000 00060000 000E0000 000C0000 000C0000
00180000 00180000 00100000 00300000 00200000 00200000 00400000
00C00000 00C00000 01800000 01800000 03000000 03000000 06000000
06000000 0C000000 1C000000 18000000 38000000 30000000 70000000
60000000 C0000000 00000000 00000000 >getsymb
213< 0000 0000 2000 1800 0C00 0400 0400 0C00 1800 3000 6000 8000 0000
0000 >getsymb
212< 0000 0000 0800 3000 6000 C000 C000 C000 6000 3000 1000 0800 0000
0000 >getsymb
211< 0000 0000 0040 1020 0810 0C18 0C18 1818 3010 4030 8060 0040 0000
0000 >getsymb
210< 00000000 00000000 00060000 04180000 18300000 30600000 60C00000
C0C00000 C0C00000 C0C00000 C0400000 40200000 20000000 00000000
00000000 >getsymb
209< 000000000000 000000000000 FFF000000000 0FFFC0000000 000FFFFE0000
000007FFF000 000000000000 000000000000 >getsymb
208< 00000000 00000000 FFC00000 7FFF8000 007FE000 00000000 00000000
>getsymb
204< 00000000 00000000 00000200 00038C00 0007F800 001CF000 00380000
00200000 00000000 00000000 00080000 00060000 00030000 00078000
0007C000 000CE000 001CF000 00303000 00303E00 003FF800 00FF1800
00400C00 00C00C00 01800600 01800600 03000300 06000300 1C000380
30000380 60000380 E0000700 00000000 00000000 >getsymb
203< 00000000 00000000 00300000 00180000 000E0000 00038000 0003C000
0000E000 0000F000 00001800 00060800 00030000 00038000 00078000
000EC000 000C6000 00182000 00302000 00703000 01FC3000 00FFF800
01807F80 03800E00 07000200 06000200 0E000300 1C000100 38000180
600000C0 C0000060 C0000060 00000000 00000000 >getsymb
201< 00000000 00000000 80180600 E03C0700 E07C0700 703C0E00 00000000
00000000 >getsymb
200< 00000000 00000000 00100000 00180000 E00C0000 30060000 18060000
0C070000 0E018000 0E01C000 0780C000 03806000 00E07000 00E03000
00703800 00303800 00607000 00606000 01C0C000 0180C000 03018000
06070000 0C060000 380C0000 30080000 00100000 00200000 00600000
00000000 00000000 >getsymb
199< 00000000 00000000 00010000 00620000 00C60000 018C0000 03180000
06180000 0C300000 18600000 10600000 70C00000 60C00000 E1C00000
61C00000 30E00000 38600000 18300000 1C100000 0E080000 070C0000
03060000 01C30000 00C30000 00438000 00418000 00008000 00000000
00000000 >getsymb
197< 00000000 00000000 00003800 0000F000 0C038000 3F870000 70FE0000
603C0000 C0000800 80001000 00003000 03802000 0FE06000 38F06000
701CC000 600F8000 60078000 00030000 00000000 00000000 >getsymb
196< 00000000 00000000 0001C000 001F8000 00380000 00600000 00600000
00400000 00400000 00C00000 00C00000 00C00000 00C00000 00CC0000
00980000 01F00000 0F800000 0D800000 01800000 01800000 01800000
01800000 01800000 01800000 01800000 01800000 01800000 01800000
01800000 01800000 03000000 02000000 06000000 8C000000 78000000
00000000 00000000 >getsymb
195< 00000000 00000000 00000400 00001800 0003F000 00078000 00060000
00040000 000C0000 000C0000 600C0000 F00C0000 300C0000 18180000
18180000 18180000 08100000 08300000 0C300000 0C600000 0C600000
0C400000 0DC00000 07800000 07000000 02000000 00000000 00000000
>getsymb
194< 00000000 00000000 03018000 0F07C000 188C6000 10C83000 20701800
60300C00 40200600 C0200300 C0000300 C0000300 40000300 40000300
60000300 60000600 20000C00 30000800 10001800 18003000 0C006000
0C00C000 06018000 03070000 010C0000 00980000 00F00000 00600000
00400000 00400000 00000000 00000000 >getsymb
192< 00000000 00000000 00C00000 00E00000 00C00000 00000000 00000000
00000000 00C00000 00C00000 00600000 00600000 00C00000 01800000
03000000 06000000 1C000000 38000000 60000000 60040000 C0060000
C0020000 C0010000 E0030000 70060000 380C0000 1C3C0000 0FF80000
07E00000 00000000 00000000 >getsymb
191< 00000000 00000000 06000000 1F000000 318C0000 609F0000 60B38000
60E18000 40418000 C040C000 C000C000 C000C000 C000C000 C0018000
40030000 60060000 600C0000 30180000 30100000 18300000 18600000
08400000 0CC00000 0C800000 05800000 03000000 03000000 02000000
00000000 00000000 >getsymb
186< 00000000 00000000 00060000 001C0000 00F80000 01F00000 01800000
03800000 03800000 07000000 07000000 07000000 07000000 07000000
07000000 07000000 07000000 07000000 07000000 07000000 07000000
0F000000 0F000000 0F000000 0F000000 0F000000 0F000000 0F000000
0E000000 0E000000 0E000000 0E000000 0E000000 0E000000 0E000000
8C000000 FC000000 F8000000 30000000 00000000 00000000 >getsymb
185< 00000000 00000000 00004000 7C003000 FF003000 CF806000 CDC0C000
0CFF0000 0C1F0000 0C030000 0C030000 0C030000 0C030000 0C010000
0C018000 3800D000 30006000 00000000 00000000 >getsymb
184< 00000000 00000000 1FC00000 7FFC0000 EE0F0000 83038000 0301C000
0180C000 01806000 01802000 01803000 01803000 01803000 01803000
01803000 01803000 01806000 0180E000 0181C000 018F8000 03FC0000
0FE00000 03000000 03000000 03000000 03000000 06000000 06000000
06000000 06000000 06000000 06000000 00000000 00000000 >getsymb
183< 00000000 00000000 000FF800 003FFE00 00F00700 03C00180 0F000080
07000000 03C00000 01E00000 00780000 001E0000 00070000 0001C000
0000E000 00007000 0000F800 0001C000 00038000 00060000 000C0000
00380000 00700000 00E00000 01C00000 03800080 0F000180 1C000180
38000180 70000300 E0000300 FC000600 7FFC0C00 03FFF800 0003F000
00000000 00000000 >getsymb
182< 0000 0000 0F00 3F80 40E0 8070 0038 0018 000C 000C 000C 000C 038C
0FEC 1C3C 301C 201C 600C 600C 600C 600C 6008 6018 6030 3060 1FC0 0F80
0000 0000 >getsymb
181< 00000000 00000000 0600C000 0600C000 0600C000 0600C000 06008000
06018000 06018000 0E018000 0E018000 0E018000 0E038000 0E038000
0E038800 1E079800 1B0DF000 19F8E000 10F04000 30000000 30000000
30000000 30000000 60000000 60000000 40000000 80000000 00000000
00000000 >getsymb
180< 00000000 00000000 08000000 0C030000 0C038000 00000000 00000000
00000000 00000000 04030000 0C030000 0C030000 18030000 38030000
30030000 70060300 F0060600 301C0C00 301C0C00 303C1800 186C3000
1C6C6000 0CCC6000 078CC000 030D8000 000F0000 000E0000 003C0000
007C0000 00CC0000 018C0000 030C0000 030C0000 03180000 03300000
01E00000 00C00000 00000000 00000000 >getsymb
179< 00000000 00000000 38000000 0E000000 03800000 00C00000 00600000
00300000 001C0000 00070000 0003C000 00018000 00030000 00060000
001C0000 00780000 00C00000 01800000 07000000 0E000000 18000000
00000000 000FE000 1FFF0000 7FE00000 00000000 00000000 3FFFC000
FFFF8000 00000000 00000000 >getsymb
178< 00000000 00000000 00018000 00070000 000C0000 00180000 00700000
01C00000 07000000 0C000000 38000000 60000000 C0000000 E0000000
30000000 18000000 0E000000 07000000 01C00000 00E00000 00380000
001C0000 00070000 00000000 00000000 3FFFF000 0FFFE000 00000000
00000000 0FE00000 07FF8000 001FE000 00000000 00000000 >getsymb
177< 00000000 00000000 01000000 03000000 03000000 03000000 03000000
03000000 03000000 03000000 FFF80000 FFFF0000 03000000 03000000
03000000 01800000 01800000 01800000 01800000 01800000 00000000
00000000 FFFFC000 0FFFC000 00000000 00000000 >getsymb
176< 00000000 00000000 000F0000 003FC000 3C60E000 7EC06000 C1806000
C180C000 61FF8000 7F1F0000 1E000000 00000000 00000000 >getsymb
175< 00000000 00000000 00000300 00000600 00000400 001FEC00 00783800
00E03C00 03806600 06006200 0E004300 0C00C300 0C018180 0C018180
18030180 18020180 18060180 180C0180 18080180 18180180 18300300
18700300 18E00200 09800600 0D000C00 0F000800 07003800 0780F000
0DFFC000 0C7F0000 18000000 30000000 60000000 C0000000 00000000
00000000 >getsymb
174< 00000000 00000000 00FF0000 03C1C000 0700E000 1C003000 30001000
70001800 60001800 60000C00 60000C00 C0000C00 C0000C00 C0000C00
C0000C00 C0000C00 C0000C00 C0001800 C0001800 C0001000 40003000
60006000 70004000 3801C000 1C078000 0FFE0000 03F80000 00000000
00000000 >getsymb
173< 00000000 00000000 00040000 000C0000 000C0000 000C0000 00180000
001FF000 07FFC000 3FF80000 78300000 00600000 00600000 00C00000
00FFE000 3FFFC000 FFC00000 01800000 01800000 03000000 03000000
06000000 04000000 00000000 00000000 >getsymb
172< 0000 0000 E038 6038 C030 0000 0000 >getsymb
171< 0000 0000 0600 1C00 3000 6000 4000 C000 C000 8000 0000 0000
>getsymb
170< 000000000000 000000000000 000F20000000 01F82C700000 FF803EF80000
FF00638C0000 6300630C0000 330063040000 130063060000 1B0063060000
0B00C3060000 0B00C3060000 0B00C3068000 0E0183028000 060101038000
000000030000 000000000000 000000000000 >getsymb
169< 00000000 00000000 0000F800 0003FC00 003F1E00 00F80700 03800180
06000180 1C0000C0 300000C0 3007C0C0 601FF0E0 60301820 60200C30
C0600430 C0C00030 C0C00030 C0C00030 C0C00030 C0C02030 C0402030
C0606030 6030C020 603F8060 380E01C0 3C000300 03001E00 03C03C00
00FFE000 00FF0000 00000000 00000000 >getsymb
168< 00000000 00000000 00078000 000FC000 000CE000 000C7000 007E1800
00FC0C00 07C00E00 0F800700 0C060300 1C9F8300 30B0C180 30C0C180
70C1C180 E0C38080 C0FF00C0 C0FC00C0 C1C800C0 C1CC00C0 C30E01C0
C3030180 E2038300 6201C300 30000700 30000E00 38003C00 1C007800
0E00E000 0F03C000 07FF8000 01FC0000 00000000 00000000 >getsymb
167< 00000000 00000000 00070000 003F8000 0C70C000 1EC04000 1E804000
1700C000 37018000 36030000 360C0000 661F0000 66018180 C6008300
0600C600 0600CC00 0600D800 0600F000 0600E000 0600C000 0E038000
0E0E0000 0C180000 0C200000 0C000000 0C000000 08000000 08000000
10000000 10000000 10000000 00000000 00000000 >getsymb
166< 00000000 00000000 00060000 00038000 01F0C000 07AFE000 3FBDF000
5F5C3000 4FFC3000 5FFC3000 BFFC1000 FFFC1000 7FFC1000 7FFC1800
7FF81800 7FF81800 FFF81800 FFF80800 7FF80800 7FF80800 3FF80800
00580800 00180800 00180800 00181800 00181800 00181800 00181000
00183000 00182000 00186000 00186000 0018C000 001CC000 000F8000
00078000 00000000 00000000 >getsymb
165< 0000 0000 3000 4F00 9FC0 FFE0 FFF0 7FF8 FFF8 FFF8 FFF8 FFFC 9FE4
AFE4 4FD8 7FF0 2700 1E00 0000 0000 >getsymb
164< 00000000 00000000 00F00000 01FC0000 03060000 06020000 06000000
07000000 03200000 01C00000 07F00000 1C1C0000 30060000 60030000
C0018000 C0018000 C0018000 E0070000 700C0000 1E780000 07E00000
03F00000 06380000 081C0000 000C0000 00060000 00030000 00018000
0400C000 0400C000 06018000 03038000 01FE0000 00F00000 00000000
00000000 >getsymb
163< 00000000 00000000 00018000 0007C000 001C6000 00303000 00601800
00600800 00400000 00C00000 00C00000 00C00000 00C00000 00C00000
00C00000 00C00000 1FC00000 03FE0000 01FFC000 01C00000 01800000
0180E000 03013000 07001800 06001800 0C001000 18002000 3FC0E000
F07FC000 601F0000 00000000 00000000 >getsymb
162< 00000000 00000000 00180000 00300000 00200000 00600000 00600000
00600000 00600000 01F00000 07F80000 0CCC0000 08C40000 18C00000
30C00000 70800000 F1801000 31803000 31806000 31804000 3180C000
31818000 19818000 19830000 0D8E0000 07980000 01F00000 01800000
01000000 01000000 03000000 03000000 02000000 00000000 00000000
>getsymb
161< 0000 0000 3C00 6600 6300 4180 C180 C080 C080 E180 FF00 4C00 6000
3000 0000 0000 >getsymb
160< 00000000 00000000 00C00000 00C00000 00C00000 00C00000 00C00000
00C00000 00C00000 00C00000 01C00000 E1C00000 FFE00000 07FC0000
01CF8000 01800000 01800000 01800000 01800000 01800000 01800000
01800000 01800000 01800000 01800000 01800000 01800000 01800000
01800000 01000000 00000000 00000000 >getsymb
159< 00000000 00000000 01000000 0300C000 0300C000 00006000 00000000
00000000 00000000 00000000 00008000 04008000 0E018000 3B018000
63010000 43010180 C3030300 02030600 06030400 06030C00 06030C00
06070800 06059800 060D9800 06189000 03F0F000 01E06000 00000000
00000000 >getsymb
158< 00000000 00000000 00100000 00780000 01CE0000 03870000 06018000
0400C000 08006000 00002000 00000000 00000000 00000000 00004000
00004000 0C018000 1E018000 33030000 63030180 C3030300 03030600
06030600 06030400 06070C00 06070C00 06051800 06091800 07191000
03F1F000 01E0E000 00000000 00000000 >getsymb
157< 00000000 00000000 01800000 00C00000 00F00000 00300000 00180000
000C0000 00060000 00000000 00000000 00000000 00008000 00018000
00018000 3C018000 7E018000 43018180 C3010300 03030600 06030600
06030C00 06070C00 06070C00 06059800 060D9800 07089000 03F8F000
01F06000 00000000 00000000 >getsymb
156< 00000000 00000000 00007000 0001C000 00030000 000E0000 00380000
00200000 00E00000 00000000 00000000 00004000 0E00C000 1E00C000
13018000 33018000 63030000 43030180 C3030300 03030600 06030600
06070C00 06070C00 060F0C00 06099800 06199800 07309000 03E0F000
01C06000 00000000 00000000 >getsymb
155< 00000000 00000000 00C0C000 01E38000 11370000 0E1E0000 04000000
00000000 00000000 00F00000 03FC0000 071E0000 0E070000 38018000
7801C600 D801EC00 1801B800 18018000 18018000 0C018000 0C018000
06010000 06030000 03060000 01FC0000 00F00000 00000000 00000000
>getsymb
154< 00000000 00000000 0C040000 0C0E0000 100E0000 00000000 00000000
00000000 00000000 00F00000 03FC0000 079E0000 0F070000 3F038000
7203C600 C6017C00 06013000 06018000 0E018000 0C018000 0C010000
06030000 06060000 030C0000 01F80000 00F00000 00000000 00000000
>getsymb
153< 00000000 00000000 00C00000 00600000 00F00000 01BC0000 030E0000
06070000 18018000 1000E000 00000000 00F00000 03FC0000 079E0000
1F070000 3C038000 7803C600 D8016C00 18013800 18018000 18018000
0C018000 0C010000 06030000 06060000 030C0000 01F80000 00F00000
00000000 00000000 >getsymb
152< 00000000 00000000 06000000 07000000 01C00000 00780000 003C0000
00070000 00000000 00000000 00F00000 03FC0000 071E0000 1C030000
38038000 7801C600 D8016C00 18013800 18018000 18018000 08018000
0C010000 0C030000 06060000 030C0000 01F80000 00F00000 00000000
00000000 >getsymb
151< 00000000 00000000 00010000 00020000 000C0000 00180000 00300000
00E00000 03800000 00000000 00000000 00000000 00000000 00000000
00F00000 03FC0000 079C0000 1C060000 38030000 7801C600 D8016C00
18013800 18018000 18018000 1C018000 0C030000 0C020000 0C060000
060C0000 07F80000 01F00000 00000000 00000000 >getsymb
150< 00000000 00000000 00000600 00E00C00 01B83800 010E6000 0603C000
0C000000 00000000 00000000 00000000 00000000 00000000 0F1C0000
3FBF0000 31B38000 61E18000 C1C18180 81818180 01818300 01818300
01818600 03018600 07018400 06018C00 0600CC00 06007800 06003000
00000000 00000000 >getsymb
149< 0000 0000 1800 3830 3878 0070 0000 0000 0000 0000 0C00 1800 1800
3000 3000 7008 F018 B030 3060 3060 30C0 30C0 3180 3180 1B00 1E00 1C00
0800 0000 0000 >getsymb
148< 0000 0000 0300 0780 0D80 18C0 3040 2060 2030 4030 0018 0000 0C00
1800 1800 3000 3000 7008 F018 B030 3060 3060 30C0 30C0 3180 3180 1B00
1E00 1C00 0800 0000 0000 >getsymb
147< 0000 0000 8000 E000 7000 1800 0C00 0700 0780 0000 0000 0000 0C00
1800 1800 3000 3000 7008 F018 B030 3060 3060 30C0 30C0 3180 3180 1B00
1E00 1C00 0800 0000 0000 >getsymb
146< 0000 0000 0018 0030 0060 01C0 0300 0E00 1C00 0000 0000 0000 0C00
1800 1800 3000 3000 7008 F018 B030 3060 3060 30C0 30C0 3180 3180 1B00
1E00 1C00 0800 0000 0000 >getsymb
145< 00000000 00000000 301C0000 380C0000 30000000 00000000 00000000
00000000 00000000 03E00000 0FF00000 1C180000 30180000 60180000
40300000 C0600000 C0C00000 C7800000 FE001800 F0001000 60003000
30006000 3000C000 18018000 0C030000 0C060000 061C0000 03F80000
00E00000 00000000 00000000 >getsymb
144< 00000000 00000000 03E00000 0FF80000 300E0000 20030000 40010000
C0000000 00000000 00000000 03E00000 07300000 0C180000 180C0000
300C0000 200C0000 60380000 60E00000 67C00000 FE001800 60003000
60006000 3000C000 30018000 10030000 18060000 0C0C0000 06380000
03F00000 01C00000 00000000 00000000 >getsymb
143< 00000000 00000000 38000000 0C000000 07000000 01800000 00C00000
00400000 00000000 00000000 01C00000 07E00000 0C300000 18180000
30080000 30380000 60E00000 C3800000 CE000000 F8000C00 F0001800
60003000 30006000 3000C000 18018000 08030000 0C060000 060C0000
03F80000 01E00000 00000000 00000000 >getsymb
142< 00000000 00000000 001C0000 00780000 00C00000 00800000 03000000
02000000 00000000 00000000 07800000 1FC00000 30600000 60300000
60300000 40600000 C1C00000 C7800000 9E000000 F0000800 C0001800
C0003000 60006000 2000C000 30018000 18030000 080E0000 0C180000
07F00000 03C00000 00000000 00000000 >getsymb
141< 00000000 00000000 03E00000 0FF00000 18180000 30080000 60000000
60003000 40006000 C000C000 C0008000 C0018000 C0018000 C0030000
40030000 60020000 30060000 1F0C0000 07F80000 00E00000 00C00000
00400000 00780000 001C0000 00040000 04060000 06060000 03060000
030C0000 01F80000 00000000 00000000 >getsymb
140< 00000000 00000000 00F00000 01F80000 02880000 00880000 00D80000
00700000 00000000 00000000 01E08000 03F98000 060D8000 0C070000
18060000 10060000 30060000 70060000 E0060600 E00C0400 600C0C00
600C1800 201C1000 30163000 30366000 18366000 18624000 0FC3C000
07818000 00000000 00000000 >getsymb
139< 00000000 00000000 00004000 00008000 0E030000 339E0000 40F00000
00000000 00000000 00000000 01E08000 03F98000 060D8000 0C070000
18060000 10060000 30060000 70060000 E0060600 E00C0400 600C0C00
600C1800 201C1000 30163000 30366000 18366000 18624000 0FC3C000
07818000 00000000 00000000 >getsymb
138< 00000000 00000000 00080000 040C0000 1C0C0000 18000000 00000000
00000000 00000000 01E08000 03F98000 060D8000 0C070000 18060000
10060000 30060000 70060000 E0060600 E00C0400 600C0C00 600C1800
201C1000 30163000 30366000 18366000 18624000 0FC3C000 07818000
00000000 00000000 >getsymb
137< 00000000 00000000 00F00000 031C0000 0E060000 18038000 00008000
00000000 00000000 01E08000 03F98000 060D8000 0C070000 18060000
10060000 30060000 70060000 E0060600 E00C0400 600C0C00 600C1800
201C1000 30163000 30366000 18366000 18624000 0FC3C000 07818000
00000000 00000000 >getsymb
136< 00000000 00000000 10000000 0C000000 04000000 06000000 03000000
01800000 00000000 00000000 01E08000 03F98000 060D8000 0C070000
18060000 10060000 30060000 70060000 E0060600 E00C0400 600C0C00
600C1800 201C1000 30163000 30366000 18366000 18624000 0FC3C000
07818000 00000000 00000000 >getsymb
135< 00000000 00000000 00030000 000E0000 00180000 00300000 00400000
00000000 00000000 00000000 01E08000 03F98000 060D8000 0C070000
18060000 10060000 30060000 70060000 E0060600 E00C0400 600C0C00
600C1800 201C1000 30163000 30366000 18366000 18624000 0FC3C000
07818000 00000000 00000000 >getsymb
134< 00000000 00000000 38003800 38003800 30001800 00000000 00000000
20000000 60001800 60001800 60001800 C0001800 C0001800 C0001800
C0003000 C0003000 C0006000 C0006000 C000C000 C001C018 C001C030
C003C060 6006C060 600C60C0 30186180 3C303300 0FE01E00 07C00C00
00000000 00000000 >getsymb
133< 00000000 00000000 0F878000 1FCFC000 307CF800 303C3C00 30300C00
30300600 30300300 30300300 30300300 30300300 30300700 60300E00
C031FC00 C073F800 C0FF0000 C0FF0000 C0C30000 C0C30000 C0C18000
C0C0C000 C0C0E000 C0C07000 E0C03000 E0C03800 70C00E00 30C00E00
30C00300 30C00300 38C00380 1CC00180 0F8000C0 0F000040 00000000
00000000 >getsymb
132< 00000000 00000000 0F000100 1F800100 30E00300 30700300 30700300
30700300 20580300 204C0300 20CC0380 60CC01C0 E0CE00C0 E0C700C0
C0C300C0 C0C300C0 C0C381C0 C0C1C380 C0C0C300 C0C0C300 C0C0C300
C0C0C300 C0C06300 C0C06300 C0C03300 C0C03300 C0C03F00 C0C01F00
C0C00F00 C1C00F00 FF800F00 3F000600 00000000 00000000 >getsymb
131< 0000 0000 000C 0018 0030 00E0 0380 0000 1FC0 09FC 181C 1000 3000
2000 7F00 6FE0 E000 C000 C000 8000 C000 F000 3E00 07F0 0000 0000
>getsymb
130< 00000000 00000000 003FC000 00FFE000 07F03000 0FC03800 1F000C00
3B000E00 33000000 73000000 63000000 63000000 C3000000 C3000000
C3000000 C3000000 C3000000 C3000000 C3000000 C3000000 C3000000
C3000000 C3000000 C3000000 C6000000 CC000200 7C000C00 3C001C00
07003000 03807000 01FFE000 00FF8000 00000000 00000000 >getsymb
129< 00000000 00000000 000F0000 001F8000 0030C000 0030C000 0010C000
001F0000 00000000 00000000 000C0000 000E0000 000E0000 000F0000
00338000 00338000 0040C000 00C0C000 00FFC000 00FFE000 0101E000
03003000 04003800 0C003800 10000C00 30000C00 60000E00 60000E00
C0000600 C0000700 00000000 00000000 >getsymb
128< 00000000 00000000 00C00000 00E0E000 00E0E000 0070F000 00000000
00000000 00000000 00000000 00030000 00030000 00038000 0003C000
000CE000 000CF000 00383000 00303800 0037FC00 003FBC00 00C00C00
00C00E00 03000700 03000300 0C000380 0C000180 3C0000C0 380000C0
E0000060 C0000060 00000000 00000000 >getsymb
126< 00000000 00000000 00001000 00003000 00004000 1C018000 3E030000
63060000 819C0000 00F00000 00600000 00000000 00000000 >getsymb
125< 0000 0000 0780 1FE0 3870 0030 0030 0030 0030 0030 0030 0070 0060
0040 0060 0030 0018 001C 0070 0060 0040 00C0 00C0 00C0 00C0 00C0 00C0
0060 0020 0030 C030 7060 1DE0 0F80 0000 0000 >getsymb
124< 0000 0000 6000 6000 6000 6000 6000 6000 6000 6000 6000 6000 6000
6000 6000 6000 6000 C000 C000 C000 C000 C000 C000 C000 C000 C000 C000
C000 C000 4000 0000 0000 >getsymb
123< 00000000 00000000 00F80000 03FC0000 07020000 06000000 0C000000
0C000000 0C000000 0C000000 0C000000 0C000000 18000000 18000000
70000000 E0000000 38000000 0C000000 0C000000 0C000000 0C000000
0C000000 0C000000 0C000000 18000000 18000000 30000000 30000000
30040000 183C0000 1CF00000 07800000 00000000 00000000 >getsymb
122< 00000000 00000000 03020000 078E0000 0CFC0000 186C0000 30080000
30180000 60300000 C0306000 0060C000 0040C000 00C18000 00818000
01830000 01030000 03060000 078C0000 0CF80000 18300000 00000000
00000000 >getsymb
121< 00000000 00000000 0600C000 0600C000 0C00C000 1C00C000 3C018000
38038000 78038000 D8031800 18031000 18073000 180F2000 181B6000
1C334000 0E62C000 07C78000 00070000 00060000 000E0000 001E0000
00360000 00660000 00C60000 01860000 01860000 01840000 018C0000
00F80000 00700000 00000000 00000000 >getsymb
120< 00000000 00000000 0C000000 1E030000 1B078000 330C8000 31180000
61B00000 C1B01800 01A03000 01E02000 00C06000 00C04000 00C0C000
01C08000 63418000 36670000 3C3E0000 18180000 00000000 00000000
>getsymb
119< 00000000 00000000 18004000 38084000 6C186000 4C186000 CC186040
08183980 18183F00 18183600 18183000 18383000 18683000 18682000
0C4C2000 0ECC6000 0386C000 01838000 00000000 00000000 >getsymb
118< 00000000 00000000 18180000 3C1C0000 6C0C0000 4E0E0000 C6073000
0607E000 0606C000 0C020000 0C030000 0C030000 0C030000 0C030000
0C060000 040C0000 06180000 06300000 03E00000 01800000 00000000
00000000 >getsymb
117< 00000000 00000000 00004000 0E00C000 1E00C000 13018000 33018000
63030000 43030000 C3030300 03030600 06030600 06070C00 06070C00
060F0C00 06099800 06199800 07309000 03E0F000 01C06000 00000000
00000000 >getsymb
116< 00000000 00000000 00100000 00300000 00300000 00700000 00F00000
01B00000 01300000 03300000 06200000 06200000 0C600000 18600000
18400000 30400000 60400000 C6460000 0F4C0000 11580000 10F00000
1FC00000 0CC00000 00800000 00800000 00800000 00800000 00800000
00800000 00000000 00000000 >getsymb
115< 00000000 00000000 00C00000 01C00000 07600000 1C600000 30300000
60304000 C010C000 00198000 00198000 001B0000 000F0000 000E0000
000C0000 00180000 00300000 00600000 18C00000 0F800000 07000000
00000000 00000000 >getsymb
114< 00000000 00000000 0C100000 1E300000 33F00000 31900000 60300000
60600000 C0410000 80C20000 01860000 030C0000 03180000 03180000
03300000 06300000 06200000 06600000 06400000 06C00000 07800000
03000000 00000000 00000000 >getsymb
113< 00000000 00000000 00004000 00FCC000 03FEC000 0F06C000 3C03C200
78018600 D8018400 10038C00 30078C00 30058800 300D9800 30099800
30119800 38333000 18633000 0FC23000 07062000 00066000 00064000
000CC000 000CC000 000D8000 000D0000 000D0000 000F0000 000E0000
00060000 00000000 00000000 >getsymb
112< 00000000 00000000 00380000 0C7C0000 1EC60000 36C20000 67830000
67030180 C6010300 06018600 06018400 0E018C00 0C008800 0C00D800
0C007000 0C002000 0C000000 0C000000 0C000000 18000000 18000000
18000000 18000000 18000000 18000000 18000000 18000000 18000000
10000000 10000000 00000000 00000000 >getsymb
111< 00000000 00000000 00F00000 03FC0000 079E0000 1F070000 3C038000
7803C000 D8016C00 18013800 18018000 18018000 0C018000 0C010000
06030000 06060000 030C0000 01F80000 00F00000 00000000 00000000
>getsymb
110< 00000000 00000000 0E0E0000 1F1F0000 33B30000 21A18000 61E18000
C1C18180 81818180 01818300 03018300 03018600 07018600 06018400
0E018C00 0C00CC00 0C007800 0C003000 00000000 00000000 >getsymb
109< 000000000000 000000000000 000000E00000 060001F00000 0F1E01300000
19BF03180000 31E382180000 31C0C6180000 6180C4180000 C180CC183000
8180C8186000 0180D8186000 0180F018C000 0180E0198000 0380C0190000
0300C01B0000 0601800E0000 0601800E0000 060180040000 060180000000
000000000000 000000000000 >getsymb
108< 0000 0000 0700 0F80 08C0 18C0 18C0 18C0 30C0 30C0 30C0 3080 3180
3300 3600 3600 3C00 3800 3008 7018 F030 B020 3060 3040 30C0 30C0 3080
1980 1F00 0E00 0000 0000 >getsymb
107< 00000000 00000000 00300000 00F80000 018C0000 018C0000 030C0000
030C0000 03180000 03300000 03600000 03C00000 03800000 03000000
07000000 0F000000 1B1E0000 323F0000 66618000 C6C18600 86C30C00
078E1800 07383000 07E03000 07806000 06C06000 0C60C000 0C20C000
0C318000 18330000 181E0000 180C0000 00000000 00000000 >getsymb
106< 0000 0000 0080 0300 0300 0000 0000 0600 0600 0600 0E00 1E00 1600
3600 6610 C630 8660 06C0 0780 0700 0600 0E00 1E00 3600 6600 6600 C600
C600 C600 C600 C600 C600 CC00 CC00 CC00 7800 3000 0000 0000 >getsymb
105< 0000 0000 0600 0600 0200 0000 0000 0000 0000 0000 0000 0C00 1800
1800 3000 3000 7008 F018 B030 3060 3060 30C0 30C0 3180 3180 1B00 1E00
1C00 0800 0000 0000 >getsymb
104< 00000000 00000000 01E00000 03300000 03180000 02180000 06180000
06180000 06180000 06300000 06200000 06600000 06C00000 07800000
07000000 0E000000 1E000000 36000000 66000000 C6000300 06380600
067C0600 06C60C00 07830C00 07030C00 06031800 06031800 06031800
0C033000 0C033000 1801E000 1800C000 00000000 00000000 >getsymb
103< 00000000 00000000 00300000 00F88000 038D8000 06070000 0C030000
18070000 38070000 780F0000 D81B0600 98330C00 18331800 18633000
0CC36000 0783C000 03078000 000F0000 003B0000 00630000 00C30000
01830000 03060000 06060000 06060000 060C0000 06180000 03F00000
01E00000 00000000 00000000 >getsymb
102< 0000 0000 0380 07C0 0CC0 08C0 18C0 18C0 1980 1980 1900 1B00 1A00
1C00 1C00 1800 3800 3800 5800 D818 1C70 1FC0 1E00 1B00 1980 18C0 18C0
1860 1860 1860 1860 1860 1860 1860 1860 18C0 18C0 0C80 0F80 0700 0000
0000 >getsymb
101< 00000000 00000000 03E00000 0FF80000 1C1C0000 300C0000 200C0000
600C0000 60380000 40E00000 C7800000 DE000400 F0000C00 E0001800
60003000 30006000 3000C000 18018000 08030000 0C060000 079C0000
03F80000 00E00000 00000000 00000000 >getsymb
100< 00000000 00000000 00004000 00004000 00004000 0000C000 0000C000
0000C000 0000C000 00018000 00018000 00018000 01E18000 07F18000
0C1B8000 181F0000 180F0000 30060000 70060000 F0020300 B0060600
30060400 300E0C00 300B0C00 101B1800 18319800 0C61B000 07C1F000
0380E000 00000000 00000000 >getsymb
99< 00000000 00000000 01F00000 07380000 0E0C0000 0C040000 18000000
30000000 70000000 F0001000 30003000 30006000 30004000 3000C000
30018000 18018000 18030000 0E0E0000 07380000 01F00000 00000000
00000000 >getsymb
98< 00000000 00000000 00600000 00F00000 01980000 010C0000 030C0000
030C0000 03180000 02100000 06300000 06600000 07C00000 07000000
06000000 0E000000 1E000000 36000000 66008000 C600C300 0601FE00
0601B800 02018000 03018000 03018000 01010000 01830000 00C60000
007C0000 00380000 00000000 00000000 >getsymb
97< 00000000 00000000 01E08000 03F98000 060D8000 0C070000 18060000
10060000 30060000 70060000 E0060600 E00C0400 600C0C00 600C1800
201C1000 30163000 30366000 18366000 18624000 0FC3C000 07818000
00000000 00000000 >getsymb
96< 0000 0000 1800 3000 3000 6000 6000 C000 C000 C000 C000 6000 3000
1800 0400 0000 0000 >getsymb
95< 00000000 00000000 FF800000 3FFFFF00 00FFFF80 00000000 00000000
>getsymb
94< 00000000 00000000 00700000 01F80000 038C0000 06060000 1C030000
3001C000 6000E000 C0003000 00001800 00000C00 00000000 00000000
>getsymb
93< 0000 0000 3FC0 FFC0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0
00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0 00C0
00C0 00C0 0180 0180 3F80 7F00 0000 0000 >getsymb
92< 00000000 00000000 E0000000 70000000 30000000 18000000 18000000
0C000000 0C000000 0E000000 06000000 06000000 03000000 03000000
01800000 01800000 00C00000 00C00000 00400000 00600000 00600000
00200000 00300000 00180000 00180000 00180000 000C0000 000E0000
00060000 00070000 00038000 0000C000 00000000 00000000 >getsymb
91< 0000 0000 7800 7F00 63C0 6000 6000 6000 6000 6000 6000 6000 6000
6000 6000 6000 6000 6000 6000 6000 C000 C000 C000 C000 C000 C000 C000
C000 E000 7C00 7F00 0000 0000 >getsymb
90< 00000000 00000000 0C01F000 3E07B000 630C2000 41F86000 8070C000
00018000 00030000 00060000 00040000 000C0000 00180000 00180000
00300000 07E00000 00FC0000 00C00000 00800000 01800040 030000C0
03000180 06000180 06000300 0C780600 09F80600 1B0F0C00 1E039800
3C00F000 70007000 00000000 00000000 >getsymb
89< 00000000 00000000 00002000 00002000 20002000 70006000 C8006000
CC006000 CC00C000 8C00C000 0C00C000 0C00C000 0C00C000 0C00C000
0C01C000 0C01C000 0403C000 0602C000 0602C600 0604CC00 021CD800
03F0F000 01C0E000 0001C000 0003C000 0006C000 000CC000 0018C000
0030C000 00608000 00418000 00C30000 00860000 008C0000 00D80000
00700000 00000000 00000000 >getsymb
88< 00000000 00000000 1F000000 31C00C00 30601E00 30303300 30387100
18186000 0C18C000 000CC000 000C8000 000D8000 000F8000 000F0000
000F0000 00070000 00060000 00060000 00060000 00060000 00060060
000600C0 00060080 000F0180 400F0180 C01F0180 C0198100 C0318300
6030C300 20604200 30C06600 1C803C00 0F801800 03000000 00000000
00000000 >getsymb
87< 000000000000 000000000000 100000000000 780000000000 4C0000000000
8C0000000000 0C0030000000 0C00300000C0 0C00600001A0 0C0060000100
0C0060000300 0C0060000300 0C00C0000300 0C00C0000300 1800C0000300
1800C0000300 1001C0000600 3001C0000600 3003C0000C00 3003C0000C30
3006C0001860 300CC00030C0 3018C00020C0 3018C0006180 3030C000C380
1830C0018700 1860C0019C00 0CC0C003F800 0F80600FE000 0600703E0000
00003FF00000 00000FC00000 000000000000 000000000000 >getsymb
86< 00000000 00000000 30000100 78000380 CC0003F0 8C000360 0C000300
0C000300 0C000300 0C000300 0C000600 0C000600 0C000600 0C000600
0C000C00 06000C00 06000C00 06000C00 06001800 06001818 06001070
060011E0 06003F80 06007C00 0300F000 0187C000 01FF0000 00780000
00000000 00000000 >getsymb
85< 000000000000 000000000000 300003000000 780003000000 CC0003000000
8C0003000000 0C0003000000 0C0003000000 180006000000 180006000000
180006000000 300006000000 300006000000 300006000000 30000C000000
30000C000000 300018000000 300018000000 300030000000 300070060000
3000700C0000 3000F0180000 1801B0180000 180318300000 0C0618600000
0F0C0CC00000 03F807800000 01F003000000 000000000000 000000000000
>getsymb
84< 00000000 00000000 0F800100 3FC00200 607E0C00 803FF800 000CE000
000C0000 000C0000 000C0000 00060000 00060000 00030000 00030000
00030000 00030000 00030000 00030000 000300C0 00030180 02030300
02030300 02030600 03030C00 01833800 01C7F000 00FF8000 003C0000
00000000 00000000 >getsymb
83< 00000000 00000000 00F00000 01FC0000 031E0000 02030000 06018000
0600C000 06004000 06004000 06008000 03000000 03800000 00F00000
003C0000 000F0000 00038000 00018000 0000C600 00006400 00006C00
00003800 00003000 20007000 70006000 9C00C000 06038000 030F0000
03FC0000 01F80000 00000000 00000000 >getsymb
82< 00000000 00000000 020FE000 033FF800 03F80C00 03C00600 07000300
0F000300 1B000300 3B000300 33000600 33000C00 63003800 6300E000
43038000 C30F0000 C3FC0000 C3FE0000 C3070018 C6070030 CC038060
4C01C0C0 7C00C080 7C006180 38006180 38006100 38003300 38003300
38001E00 38001E00 30000C00 10000000 00000000 00000000 >getsymb
81< 00000000 00000000 001FE000 007FF000 01C03800 03001C00 06000C00
0C000C00 0C000C00 18000C00 18000600 10000600 30000300 30000300
30000300 30000300 60000300 60000200 60000600 C0000600 C000060C
C000060C C0000C18 C0000C18 C0001830 603C3020 304E7060 3083C040
1C01E0C0 0E037180 03FE1F00 00FC0E00 00000000 00000000 >getsymb
80< 00000000 00000000 03FC0000 0FFF0000 3C01C000 7C007000 4C003800
8C001800 0C000C00 0C000C00 0C000C00 0C000C00 06000C00 06000800
06001800 06003000 261FE000 1FFF8030 0F800060 060000C0 06000180
06000300 06000E00 06003800 0600F000 0603C000 0C0F0000 0C3C0000
0FF00000 1FC00000 00000000 00000000 >getsymb
79< 00000000 00000000 00FE0000 01FF8000 0700C000 0E00C000 0C006000
18003000 30003000 30003000 70003000 60001800 40001C00 C0000C00
C0000C00 C0000C00 C0000600 C0000600 C0000700 C0000730 C00006E0
C0000600 60000600 60000600 30000400 30000C00 30001800 18003000
0C006000 0C00C000 06018000 060F0000 03FC0000 00F00000 00000000
00000000 >getsymb
78< 000000000000 000000000000 1E0380000000 3F07E0000000 631C38000000
81B81C000000 00F00C000000 00E006000000 00C002000000 00C003000000
00C003000000 01C003000000 018003000000 018003000000 030003000000
030003000000 030003800000 030001800000 030001800400 030001800800
030001801800 060001803000 060000806000 0C0000C0C000 0C0000C0C000
0C0000C18000 0C0000C30000 0C0000C60000 0000007C0000 000000380000
000000000000 000000000000 >getsymb
77< 000000000000 000000000000 070180000000 1F83C0000000 7186600F0000
E0C6703F8000 00C43030C000 00CC1860C000 00CC18606000 00C808C06000
00F80CC02000 00F006803000 00F007803000 00E007803000 00E003803000
00E003003000 00E003003000 00E003006000 00E003006000 00E003006060
00E0030060C0 00E0030060C0 00E003006180 00E003006180 00E003006100
00E003006300 00C001006200 00C001006600 00C001003C00 00C001001800
000000000000 000000000000 >getsymb
76< 00000000 00000000 00300000 00780000 004C0000 00CC0000 00CC0000
00CC0000 00CC0000 00D80000 00F00000 00E00000 01C00000 03C00000
0EC00000 38C00000 60C00000 C0C00000 00C00000 00C00300 00C00600
00C00C00 00C00C00 00C01800 0FE01800 1FF03000 31B83000 331C6000
1E07C000 0C038000 00000000 00000000 >getsymb
75< 00000000 00000000 18300060 3C7800C0 664C0300 43CC0600 838C1C00
010C3800 000C6000 000CC000 000F8000 000F0000 00098000 0018C000
00186000 00186000 00183000 00183000 00183018 00301830 00301830
00201860 00600C60 20C00CC0 218004C0 330006C0 1E000380 0C000300
00000000 00000000 >getsymb
74< 00000000 00000000 00040000 180E0000 3E1B0000 23198000 63318000
C1B18000 C1E18000 80C18000 80018000 00030000 00030000 00060000
00060000 00060000 00060000 00062000 00066000 0006C000 00078000
00070000 00060000 000C0000 001C0000 003C0000 006C0000 00CC0000
018C0000 030C0000 060C0000 0C0C0000 0C0C0000 0C0C0000 0C0C0000
0C180000 0C180000 06180000 03980000 03F80000 00F00000 00000000
00000000 >getsymb
73< 00000000 00000000 000C0000 1C3C0000 36660000 63C60000 41860000
80060000 80060000 00060000 00060000 000C0000 000C0000 000C0000
000C0000 000C0000 000C0000 00180000 00183000 00186000 00186000
0018C000 80198000 801F0000 C03C0000 60700000 3FC00000 1F800000
00000000 00000000 >getsymb
72< 000000000000 000000000000 000001800000 000007C00000 1C100C600000
3E3808600000 67D818600000 439818600000 801818400000 801818C00000
001818800000 001819800000 00181F000000 00181C000000 001878000000
001BF0000000 007E30000000 00F830000000 019830000000 031860030000
061860060000 0618600C0000 0C18600C0000 181860180000 181860180000
181060300000 183030300000 186030600000 0FC018600000 070008C00000
00000F800000 000007000000 000000000000 000000000000 >getsymb
71< 00000000 00000000 00180000 00FF0000 03C78000 0600E000 0C003000
18001000 30000000 30000000 60000000 E0000000 C0000000 C0000000
C0000000 C0000000 C0000000 60008000 60018000 30038030 30078060
180F80C0 081D8380 0C310700 07E31C00 03C37000 0003C000 001F8000
007F0000 01E30000 03830000 07030000 06060000 0C060000 0C040000
0C0C0000 0C180000 0C100000 06300000 03E00000 01C00000 00000000
00000000 >getsymb
70< 00000000 00000000 0801FC00 0C0FF800 3FFE0000 FFF00000 8C000000
0C000000 0C000000 04000000 06000000 06000000 03000000 0303F000
03FF8000 03FC0000 03000000 02000000 06000000 06000600 04000C00
0C001800 0C003000 0C00E000 18078000 180F0000 1FF80000 3FE00000
00000000 00000000 >getsymb
69< 00000000 00000000 03FC0000 07FF0000 1C038000 3C01C000 3000C000
30004000 30000000 38000000 1E000000 0E000000 03F00000 01F80000
03800000 0E000000 18000000 30000000 60000000 60000300 C0000600
C0000600 C0000C00 C0000C00 C0001800 70003000 3800E000 1E01C000
07FF0000 01FC0000 00000000 00000000 >getsymb
68< 00000000 00000000 07FE0000 1FFF8000 3060C000 30C06000 30C03000
19801800 19800800 0D000C00 07000400 07000600 03000300 03800300
06C00300 06600300 06300300 061C0300 0C06030C 0C038318 1C01C670
18007FC0 30003F00 3E001800 3B803000 71F0E000 C03FC000 C01F8000
00000000 00000000 >getsymb
67< 00000000 00000000 007F8000 01FFE000 07007000 0C003000 18001800
38001800 30001800 60003000 60006000 6011C000 C01F0000 C0060000
C0000000 C0000000 C0000000 C0000000 C0000000 C0000000 C00000C0
C0000180 C0000180 60000300 60000300 60000600 30000C00 38003800
1E00E000 0F03C000 01FF0000 00FE0000 00000000 00000000 >getsymb
66< 00000000 00000000 03FC0000 0FFF0000 1B818000 2300C000 0300C000
0300C000 0300C000 06018000 06038000 060F0000 041C0000 04700000
0CF00000 0C1C0000 0C0F0000 0C078000 0C00C000 0C00600C 0C003018
0C003818 0C001C30 0C000C30 0C000CE0 0C000DC0 18001F00 18001E00
30003800 30007000 F800C000 7F018000 47FF0000 00FE0000 00000000
00000000 >getsymb
65< 00000000 00000000 001E0400 007F0C00 01C1C800 0300E800 06003800
0C003000 18003000 10003000 30003000 60006000 60006000 60006000
C0006000 C0006000 C000C000 C000C030 C000C060 C000C0C0 4000C180
6000C300 2001C300 3001C600 3803C600 1802CC00 0C06CC00 06046C00
039C7800 00F83000 00000000 00000000 >getsymb
64< 00000000 00000000 001F0000 00FFE000 07C0F000 0F001800 1C000C00
18000C00 30000700 70000300 60000180 60000180 400E0180 401F8180
C030C180 C070C180 C0C04180 C1C0C180 C180C180 8300C180 C301C180
C3018100 43018300 61018300 2186EE00 30CE7C00 107C0000 18000000
0C000000 06000000 0300E000 01FFC000 007E0000 00000000 00000000
>getsymb
63< 0000 0000 1E00 3F00 6180 C0C0 8060 8020 C030 6030 1030 0020 0060
00C0 0180 0300 0200 0400 0C00 0800 1800 1000 3000 3000 1800 0000 0000
0000 1800 1C00 0000 0000 >getsymb
62< 00000000 00000000 C0000000 60000000 30000000 18000000 0C000000
06000000 03000000 01800000 00C00000 00600000 00380000 001C0000
00060000 00030000 00070000 001C0000 00700000 01C00000 07000000
0E000000 38000000 60000000 C0000000 80000000 00000000 00000000
>getsymb
61< 00000000 00000000 FFE00000 FFFFF000 001FF000 00000000 00000000
00000000 00000000 00000000 00000000 FFFFC000 FFFFC000 00000000
00000000 >getsymb
60< 00000000 00000000 00018000 000F0000 00380000 00E00000 03800000
06000000 0C000000 10000000 30000000 E0000000 60000000 20000000
30000000 18000000 0C000000 06000000 03000000 01800000 00E00000
00380000 000E0000 00038000 00000000 00000000 >getsymb
59< 0000 0000 1800 3800 3000 2000 0000 0000 0000 0000 0000 0000 0000
0000 0000 0000 0C00 0600 0600 0600 0600 1C00 3000 E000 8000 0000 0000
>getsymb
58< 0000 0000 6000 F000 C000 C000 0000 0000 0000 0000 0000 0000 0000
0000 0000 0000 0000 8000 F000 E000 0000 0000 >getsymb
57< 00000000 00000000 03F00000 0FFC0000 180E0000 38060000 60020000
C0030000 C0030000 C0030000 C0030000 C0030000 C0020000 40060000
60060000 601E0000 30340000 18640000 0FEC0000 078C0000 00080000
00080000 00180000 00180000 00100000 80300000 C0300000 70600000
1FC00000 0F000000 00000000 00000000 >getsymb
56< 00000000 00000000 007C0000 00C60000 01820000 03030000 03010000
03010000 03010000 03030000 01030000 01030000 03860000 0F8E0000
18F80000 30700000 60100000 60180000 60080000 C0080000 C0080000
C0180000 40180000 60300000 3FE00000 0FC00000 00000000 00000000
>getsymb
55< 0000 0000 0008 3078 78F8 CC98 8798 0710 0030 0030 0020 0060 0040
00C0 0080 3980 0FE0 03F8 0300 0300 0600 0600 0600 0C00 0800 1800 3000
6000 6000 4000 4000 0000 0000 >getsymb
54< 00000000 00000000 00040000 001C0000 00300000 00E00000 01800000
03000000 06000000 0C000000 08000000 18F00000 33F80000 360E0000
6C070000 78010000 60018000 60018000 C000C000 C000C000 C000C000
C0008000 C0018000 60010000 30030000 18060000 1C0C0000 0F180000
03F00000 00000000 00000000 >getsymb
53< 00000000 00000000 01FF0000 03FF0000 03000000 03000000 06000000
06000000 06000000 06000000 06000000 03C00000 00F80000 001C0000
000E0000 00030000 00030000 00030000 000E0000 E01C0000 70300000
3FE00000 0F800000 00000000 00000000 >getsymb
52< 0000 0000 8018 8018 C018 C018 C018 C018 C018 FF18 3FF8 01F8 0030
0030 0070 0060 0060 0060 00E0 00C0 00C0 0080 0080 0000 0000 >getsymb
51< 00000000 00000000 01E00000 07F00000 0E180000 0C0C0000 000C0000
000C0000 00180000 00300000 00600000 00C00000 00780000 001E0000
00070000 00018000 0001C000 0000C000 8000C000 C0018000 C0070000
7C0E0000 3FF80000 03F00000 00000000 00000000 >getsymb
50< 00000000 00000000 07800000 1FC00000 38600000 70300000 C0300000
80100000 00100000 00100000 00300000 00300000 00200000 00600000
00C00000 01800000 01000000 03000000 06000000 06000000 0F004000
1DC08000 307F8000 303E0000 00000000 00000000 >getsymb
49< 0000 0000 3000 7800 9800 1800 1800 1800 1800 1800 1800 1800 1800
3000 3000 3000 3000 3000 3000 3000 3000 3000 0000 0000 >getsymb
48< 00000000 00000000 00E00000 03F80000 0E180000 180C0000 30060000
30060000 60020000 E0030000 C0030000 C0030000 C0030000 C0030000
C0030000 C0030000 C0020000 60060000 60060000 20040000 300C0000
18180000 0C180000 07F00000 01E00000 00000000 00000000 >getsymb
47< 00000000 00000000 00040000 00060000 000C0000 000C0000 00180000
00180000 00180000 00300000 00300000 00200000 00600000 00600000
00C00000 00C00000 01800000 01800000 03000000 03000000 06000000
06000000 0C000000 0C000000 18000000 18000000 38000000 30000000
60000000 60000000 60000000 C0000000 80000000 00000000 00000000
>getsymb
46< 0000 0000 8000 C000 E000 C000 0000 0000 >getsymb
45< 0000 0000 F000 7FF0 07F0 0000 0000 >getsymb
44< 0000 0000 1000 1800 0C00 0C00 0C00 0C00 0800 1800 3000 6000 8000
0000 0000 >getsymb
43< 00000000 00000000 00100000 00100000 00300000 00300000 00300000
00300000 00300000 0033E000 3FFFC000 FFFC0000 00300000 00300000
00300000 00300000 00300000 00300000 00600000 00600000 00400000
00400000 00000000 00000000 >getsymb
42< 00000000 00000000 30180000 18300000 0C600000 04C00000 0FFE0000
3FF80000 EC600000 86200000 06300000 03180000 03000000 01000000
00000000 00000000 >getsymb
41< 0000 0000 6000 F000 1C00 0E00 0300 0180 0080 00C0 00C0 00C0 00C0
00C0 0060 0060 0060 0060 0060 0060 0060 00C0 00C0 00C0 00C0 00C0 0080
0180 0380 0300 0600 0C00 1800 7000 0000 0000 >getsymb
40< 0000 0000 0700 0E00 1800 3000 2000 6000 6000 6000 6000 6000 C000
C000 C000 C000 C000 C000 C000 C000 6000 6000 6000 6000 3000 3000 1800
1800 0C00 0600 0700 0380 0000 0000 >getsymb
39< 0000 0000 2000 3000 1800 1800 0C00 0C00 0C00 1800 3000 3000 6000
4000 8000 0000 0000 >getsymb
38< 00000000 00000000 0F000000 1F800000 30C00000 20600000 20300000
30300000 18200000 18400000 0CC00000 05800000 0F000000 1F000000
19820000 30C20000 60660000 602C0000 C0380000 C0300000 C0300000
C0780000 60680000 61CC0000 3F0C0000 1E040000 00060000 00000000
00000000 >getsymb
37< 00000000 00000000 0E03C000 3FDF8000 61F30000 60C70000 C0C60000
C0CC0000 618C0000 630C0000 3E180000 0C300000 00300000 00600000
00600000 00C00000 01C00000 03800000 0307C000 060FE000 0C3C2000
18F03000 31E03000 37603000 6CC03000 78C06000 E0F0C000 C07FC000
000F0000 00000000 00000000 >getsymb
36< 00000000 00000000 01040000 03040000 030C0000 030C0000 030C0000
037C0000 03DE0000 0F0F0000 0E0D8000 1A0C8000 360C0000 260C0000
260C0000 1E0C0000 1FCC0000 07FC0000 043C0000 040E0000 8C0F8000
CC0FC000 7C0C4000 1C0C6000 0E0CE000 0FFFC000 0CFE0000 0C180000
0C180000 0C180000 0C180000 0C180000 0C100000 08100000 08100000
08100000 00100000 00000000 00000000 >getsymb
35< 00000000 00000000 00430000 00430000 00C30000 00C30000 00C30000
00C30000 00C3F000 00FFF000 1FFF0000 3F870000 018C0000 018C0000
018C0000 018C0000 030C0000 030FC000 7FFFC000 FFF80000 06180000
06180000 06180000 0C180000 0C180000 0C100000 00000000 00000000
>getsymb
34< 0000 0000 8300 6180 30C0 10C0 10C0 10C0 3180 6300 C000 8000 0000
0000 >getsymb
33< 0000 0000 4000 6000 6000 6000 6000 6000 6000 6000 6000 6000 6000
6000 6000 6000 6000 6000 E000 C000 C000 C000 C000 C000 C000 8000 0000
0000 0000 C000 C000 4000 0000 0000 >getsymb
Hwfdict begin
  /BuildChar
  Chread
end
/Joepie Hwfdict definefont
%%EndFont Joepie\n\n"))

(defun handwrite-set-pagenumber-off ()
  "Set page numbering off."
  (setq handwrite-pagenumbering nil)
  (message "page numbering off"))

(defun handwrite-set-pagenumber-on ()
  "Set page numbering on."
  (setq handwrite-pagenumbering t)
  (message "page numbering on" ))

(provide 'handwrite)

;;; handwrite.el ends here
